/* $Id: intr.c,v 1.19 1999/04/14 17:49:09 ericjb Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

/* This file groups all the functions associated with the interrupt
   subsystem. */

#include "sema.h"

/*
 *********************************************************************
 Configures the first module in a group, <ID> to pull interrupt line,
 <priority> when the condition(s) present in <mask> happen.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_set_interrupt(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 priority,
		    SHORTSIZ16 mask)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_set_interrupt(0x%p, %d, %d, %d)\n",
		     hw, ID, priority, mask));

    error = e1432_set_interrupt_mask(hw, ID, mask);
    if (error)
	return error;

    error = e1432_set_interrupt_priority(hw, ID, priority);
    if (error)
	return error;

    return 0;
}

/*
 *********************************************************************
 This is in intr.c because the host must write to some VXI registers
 that are inaccessible from the E1432 processor.  We tell the E1432
 processor anyway, so that we can use a generic "get" function, and so
 the E1432 knows what event interrupts to save and re-interrupt on.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_set_interrupt_mask(E1432ID hw, SHORTSIZ16 ID,
			 SHORTSIZ16 value)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, error2, prio, old_mask;
    int     mod;

    TRACE_PRINTF(0, ("e1432_set_interrupt_mask(0x%p, %d, %d)\n",
		     hw, ID, value));

    /* check if channel or group */
    if (ID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error = e1432_set_interrupt_mask(hw,
					     i1432_get_chan_from_module(mn),
					     value);
	    if (error)
		return error;
	}
    }
    else
    {
	/* Set lone module */
	error = e1432_get_interrupt_mask(hw, ID, &old_mask);
	if (error)
	    return error;

	error = i1432_introff();
	if (error)
	    return error;

	if (old_mask != value)
	{
	    /* Reset new mask bits before enabling interrupt from them */
	    error = e1432_write_register(hw, ID, E1432_IRQ_RESET_REG,
					 (SHORTSIZ16)(value & ~old_mask));
	    if (error)
		goto cleanup;
	}

	/* Tell module the new interrupt mask */
	error = i1432_set_mod(hw, ID, E1432_INTERRUPT_MASK_REG,
			      E1432_CMD_SET_INTERRUPT_MASK,
			      (LONGSIZ32) value);
	if (error)
	    goto cleanup;

	/* Set up host-only interrupt config register */
	error = e1432_get_interrupt_priority(hw, ID, &prio);
	if (error)
	    goto cleanup;
	error = e1432_write_register(hw, ID, E1432_IRQ_CONFIG_REG,
				     (SHORTSIZ16)(E1432_IRQ_ENABLE | value | prio));
	if (error)
	    goto cleanup;

    cleanup:
	error2 = i1432_intron();
	if (error2)
	    return error2;

	if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 This is in intr.c because the host must write to some VXI registers
 that are inaccessible from the E1432 processor.  We tell the E1432
 processor anyway, so that we can use a generic "get" function.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_set_interrupt_priority(E1432ID hw, SHORTSIZ16 ID,
			     SHORTSIZ16 value)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, error2, mask, old_prio;
    int     mod;

    TRACE_PRINTF(0, ("e1432_set_interrupt_priority(0x%p, %d, %d)\n",
		     hw, ID, value));

    /* check if channel or group */
    if (ID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error = e1432_set_interrupt_priority
		(hw, i1432_get_chan_from_module(mn), value);
	    if (error)
		return error;
	}
    }
    else
    {
	/* Set lone module */
	error = e1432_get_interrupt_priority(hw, ID, &old_prio);
	if (error)
	    return error;

	error = i1432_introff();
	if (error)
	    return error;

	if (old_prio == 0)
	{
	    /* Reset all mask bits before enabling interrupts, if
	       interrupts were disabled before */
	    error = e1432_write_register(hw, ID, E1432_IRQ_RESET_REG,
					 (SHORTSIZ16) E1432_IRQ_MASK);
	    if (error)
		goto cleanup;
	}

	/* Tell module the new interrupt priority */
	error = i1432_set_mod(hw, ID, E1432_INTERRUPT_PRIORITY_REG,
			      E1432_CMD_SET_INTERRUPT_PRIORITY,
			      (LONGSIZ32) value);
	if (error)
	    goto cleanup;

	/* Set up host-only interrupt config register */
	error = e1432_get_interrupt_mask(hw, ID, &mask);
	if (error)
	    goto cleanup;
	error = e1432_write_register(hw, ID, E1432_IRQ_CONFIG_REG,
				     (SHORTSIZ16)(E1432_IRQ_ENABLE | value | mask));
	if (error)
	    goto cleanup;

    cleanup:
	error2 = i1432_intron();
	if (error2)
	    return error2;
	if (error)
	    return error;
    }

    return 0;
}

/*
 *********************************************************************
 Re-enable interrupts on the E1432.  After asserting an interrupt, the
 E1432 will not assert any more interrupts until this function is
 called.  This function formerly just called i1432_set_mod().  That
 worked, but was slow.  This just pokes a value to memory, and so is
 quite fast.  We can do that because this is a one-way communication
 and we're not worried about module errors.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_reenable_interrupt(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod;

    TRACE_PRINTF(0, ("e1432_reenable_interrupt(0x%p, %d)\n", hw, ID));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    if (ID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error = e1432_reenable_interrupt(hw,
					     i1432_get_chan_from_module(mn));
	    if (error)
		return error;
	}
	return 0;
    }
    else
	return e1432_write_register(hw, ID,
				    E1432_REENABLE_INTERRUPT_REG, 1);
}
